'use strict';

/* --------------------------------------------------------------
 cart_handler.js 2019-06-20
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2019 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Component for handling the add to cart and wishlist features
 * at the product details and the category listing pages. It cares
 * for attributes, properties, quantity and all other
 * relevant data for adding an item to the basket or wishlist
 */
gambio.widgets.module('cart_handler', ['hooks', 'form', 'xhr', 'loading_spinner', gambio.source + '/libs/events', gambio.source + '/libs/modal.ext-magnific', gambio.source + '/libs/modal'], function (data) {

    'use strict';

    // ########## VARIABLE INITIALIZATION ##########

    var $this = $(this),
        $body = $('body'),
        $window = $(window),
        busy = false,
        ajax = null,
        timeout = 0,
        defaults = {
        // AJAX "add to cart" URL
        addCartUrl: 'shop.php?do=Cart/BuyProduct',
        // AJAX "add to cart" URL for customizer products
        addCartCustomizerUrl: 'shop.php?do=Cart/Add',
        // AJAX URL to perform a value check
        checkUrl: 'shop.php?do=CheckStatus',
        // AJAX URL to perform the add to wishlist
        wishlistUrl: 'shop.php?do=WishList/Add',
        // Submit URL for price offer button
        priceOfferUrl: 'gm_price_offer.php',
        // Submit method for price offer
        priceOfferMethod: 'get',
        // Selector for the cart dropdown
        dropdown: '#head_shopping_cart',
        // "Add to cart" buttons selectors
        cartButtons: '.js-btn-add-to-cart',
        // "Wishlist" buttons selectors
        wishlistButtons: '.btn-wishlist',
        // "Price offer" buttons selectors
        priceOfferButtons: '.btn-price-offer',
        // Selector for the attribute fields
        attributes: '.js-calculate',
        // Selector for product property
        productOptions: '.modifier-group .modifier-content .modifier-item',
        productOptionField: '.hidden-input',
        // Selector for the quantity
        quantity: '.js-calculate-qty',
        // URL where to get the theme for the dropdown
        tpl: null,
        // Show attribute images in product images swiper (if possible)
        // -- this feature is not supported yet --
        attributImagesSwiper: false,
        // Trigger the attribute images to this selectors
        triggerAttrImagesTo: '#product_image_swiper, #product_thumbnail_swiper, ' + '#product_thumbnail_swiper_mobile',
        // Class that gets added to the button on processing
        processingClass: 'loading',
        // Duration for that the success or fail class gets added to the button
        processingDuration: 2000,
        // AJAX response content selectors
        selectorMapping: {
            buttons: '.shopping-cart-button',
            giftContent: '.gift-cart-content-wrapper',
            giftLayer: '.gift-cart-layer',
            shareContent: '.share-cart-content-wrapper',
            shareLayer: '.share-cart-layer',
            hiddenOptions: '#cart_quantity .hidden-options',
            message: '.global-error-messages',
            messageCart: '.cart-error-msg',
            messageHelp: '.help-block',
            modelNumber: '.model-number',
            modelNumberText: '.model-number-text',
            price: '.current-price-container',
            propertiesForm: '.properties-selection-form',
            quantity: '.products-quantity-value',
            ribbonSpecial: '.ribbon-special',
            shippingInformation: '#shipping-information-layer',
            shippingTime: '.products-shipping-time-value',
            shippingTimeImage: '.img-shipping-time img',
            totals: '#cart_quantity .total-box',
            weight: '.products-details-weight-container span'
        },
        page: 'product-listing'
    },
        options = $.extend(true, {}, defaults, data),
        module = {},
        mobile = $(window).width() <= 767;

    // ########## HELPER FUNCTIONS ##########

    /**
     * Helper function that updates the button
     * state with an error or success class for
     * a specified duration
     * @param   {object}        $target         jQuery selection of the target button
     * @param   {string}        state           The state string that gets added to the loading class
     * @private
     */
    var _addButtonState = function _addButtonState($target, state) {
        var timer = setTimeout(function () {
            $target.removeClass(options.processingClass + ' ' + options.processingClass + state);
        }, options.processingDuration);

        $target.data('timer', timer).addClass(options.processingClass + state);
    };

    /**
     * Helper function to set the messages and the
     * button state.
     * @param       {object}    data                Result form the ajax request
     * @param       {object}    $form               jQuery selecion of the form
     * @param       {boolean}   disableButtons      If true, the button state gets set to (in)active
     * @param       {boolean}   showNoCombiMesssage If true, the error message for missing property combination selection will be displayed
     * @private
     */
    var _stateManager = function _stateManager(data, $form, disableButtons, showNoCombiSelectedMesssage) {

        // Remove the attribute images from the common content
        // so that it doesn't get rendered anymore. Then trigger
        // an event to the given selectors and deliver the
        // attrImages object
        if (options.attributImagesSwiper && data.attrImages && data.attrImages.length) {
            delete data.content.images;
            $(options.triggerAttrImagesTo).trigger(jse.libs.theme.events.SLIDES_UPDATE(), { attributes: data.attrImages });
        }

        // Set the messages given inside the data.content object
        $.each(data.content, function (i, v) {
            var $element = $form.parent().find(options.selectorMapping[v.selector]);

            if ((!showNoCombiSelectedMesssage || v.value === '') && i === 'messageNoCombiSelected') {
                return true;
            }

            switch (v.type) {
                case 'hide':
                    if (v.value === 'true') {
                        $element.hide();
                    } else {
                        $element.show();
                    }
                    break;
                case 'html':
                    $element.html(v.value);
                    break;
                case 'attribute':
                    $element.attr(v.key, v.value);
                    break;
                case 'replace':
                    if (v.value) {
                        $element.replaceWith(v.value);
                    } else {
                        $element.addClass('hidden').empty();
                    }
                    break;
                default:
                    $element.text(v.value);
                    break;
            }
        });

        // Dis- / Enable the buttons
        if (disableButtons) {
            var $buttons = $form.find(options.cartButtons);
            if (data.success) {
                $buttons.removeClass('inactive');
                $buttons.removeClass('btn-inactive');
                $buttons.prop("disabled", false);
            } else {
                $buttons.addClass('inactive');
                $buttons.addClass('btn-inactive');
                $buttons.prop("disabled", true);
            }
        }

        if (data.content.message) {
            var $errorField = $form.find(options.selectorMapping[data.content.message.selector]);
            if (data.content.message.value) {
                $errorField.removeClass('hidden').show();
            } else {
                $errorField.addClass('hidden').hide();

                if (showNoCombiSelectedMesssage && data.content.messageNoCombiSelected !== undefined && data.content.messageNoCombiSelected) {
                    if (data.content.messageNoCombiSelected.value) {
                        $errorField.removeClass('hidden').show();
                    } else {
                        $errorField.addClass('hidden').hide();
                    }
                }
            }
        }

        $window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());
    };

    /**
     * Helper function to send the ajax
     * On success redirect to a given url, open a layer with
     * a message or add the item to the cart-dropdown directly
     * (by triggering an event to the body)
     * @param       {object}      data      Form data
     * @param       {object}      $form     The form to fill
     * @param       {string}      url       The URL for the AJAX request
     * @private
     */
    var _addToSomewhere = function _addToSomewhere(data, $form, url, $button) {
        function callback() {
            jse.libs.xhr.post({ url: url, data: data }, true).done(function (result) {
                try {
                    // Fill the page with the result from the ajax
                    _stateManager(result, $form, false);

                    // If the AJAX was successful execute
                    // a custom functionality
                    if (result.success) {
                        switch (result.type) {
                            case 'url':
                                if (result.url.substr(0, 4) !== 'http') {
                                    location.href = jse.core.config.get('appUrl') + '/' + result.url;
                                } else {
                                    location.href = result.url;
                                }

                                break;
                            case 'dropdown':
                                $body.trigger(jse.libs.theme.events.CART_UPDATE(), [true]);
                                break;
                            case 'layer':
                                jse.libs.theme.modal.info({ title: result.title, content: result.msg });
                                break;
                            default:
                                break;
                        }
                    }
                } catch (ignore) {}
                _addButtonState($button, '-success');
            }).fail(function () {
                _addButtonState($button, '-fail');
            }).always(function () {
                // Reset the busy flag to be able to perform
                // further AJAX requests
                busy = false;
            });
        }

        if (!busy) {
            // only execute the ajax
            // if there is no pending ajax call
            busy = true;

            jse.libs.hooks.execute(jse.libs.hooks.keys.shop.cart.add, data, 500).then(callback).catch(callback);
        }
    };

    // ########## EVENT HANDLER ##########

    /**
     * Handler for the submit form / click
     * on "add to cart" & "wishlist" button.
     * It performs a check on the availability
     * of the combination and quantity. If
     * successful it performs the add to cart
     * or wishlist action, if it's not a
     * "check" call
     * @param       {object}    e      jQuery event object
     * @private
     */
    var _submitHandler = function _submitHandler(e) {
        if (e) {
            e.preventDefault();
        }

        var $self = $(this),
            $form = $self.is('form') ? $self : $self.closest('form'),
            customizer = $form.hasClass('customizer'),
            properties = !!$form.find('.properties-selection-form').length,
            module = properties ? '' : '/Attributes',
            showNoCombiSelectedMesssage = e && e.data && e.data.target && e.data.target !== 'check';

        if ($form.length) {

            // Show properties overlay
            // to disable user interaction
            // before markup replace
            if (properties) {
                $this.addClass('loading');
            }

            if ($self.is('select')) {
                var price = $self.find(":selected").attr('data-price');
                $self.parents('.modifier-group').find('.selected-value-price').text(price);
            }

            var getGalleryHash = $('#current-gallery-hash').val();
            $form.find('#update-gallery-hash').val(getGalleryHash);

            var formdata = jse.libs.form.getData($form, null, true);
            formdata.target = e && e.data && e.data.target ? e.data.target : 'check';
            formdata.isProductInfo = $form.hasClass('product-info') ? 1 : 0;

            // Abort previous check ajax if
            // there is one in progress
            if (ajax && e) {
                ajax.abort();
            }

            // Add processing-class to the button
            // and remove old timed events
            if (formdata.target !== 'check') {
                var timer = $self.data('timer');
                if (timer) {
                    clearTimeout(timer);
                }

                $self.removeClass(options.processingClass + '-success ' + options.processingClass + '-fail').addClass(options.processingClass);
            }

            ajax = jse.libs.xhr.get({
                url: options.checkUrl + module,
                data: formdata
            }, true).done(function (result) {
                _stateManager(result, $form, true, showNoCombiSelectedMesssage);
                $this.removeClass('loading');

                if (result.success) {
                    var event = null,
                        url = null;

                    switch (formdata.target) {
                        case 'wishlist':
                            if (customizer) {
                                event = jse.libs.theme.events.ADD_CUSTOMIZER_WISHLIST();
                            }
                            url = options.wishlistUrl;
                            break;
                        case 'cart':
                            if (customizer) {
                                event = jse.libs.theme.events.ADD_CUSTOMIZER_CART();
                                url = options.addCartCustomizerUrl;
                            } else {
                                url = options.addCartUrl;
                            }
                            break;
                        case 'price_offer':
                            $form.attr('action', options.priceOfferUrl).attr('method', options.priceOfferMethod);
                            $form.off('submit');
                            $form.submit();

                            return;
                        default:
                            setTimeout(function () {
                                $window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());
                            }, 250);
                            break;
                    }

                    if (event) {
                        var deferred = $.Deferred();
                        deferred.done(function (customizerRandom) {
                            formdata[customizerRandom] = 0;
                            _addToSomewhere(formdata, $form, url, $self);
                        }).fail(function () {
                            _addButtonState($self, '-fail');
                        });
                        $body.trigger(event, [{ 'deferred': deferred, 'dataset': formdata }]);
                    } else if (url) {
                        _addToSomewhere(formdata, $form, url, $self);
                    }
                }

                // Check if the gallery images changed
                if (result.content.replaceGallery === true) {

                    var galleryLoading = jse.libs.loading_spinner.show($('.product-info-stage'), 9999);

                    var mainImageGallery = result.content.imageGallery;
                    var modalImageGallery = result.content.imageModal;

                    var mainGalleryWrapper = $('#image-collection-container');
                    var modalGalleryWrapper = $('#product_image_layer');

                    mainGalleryWrapper.html(mainImageGallery);
                    modalGalleryWrapper.html(modalImageGallery);

                    gambio.widgets.init($('#product_image_swiper')).done(function (arg1, arg2, arg3) {
                        var container = $('#product_image_swiper');
                        if (container.swiper() !== undefined) {
                            container.swiper().update();
                        }
                        container.trigger(jse.libs.theme.events.SLIDES_UPDATE());
                        if (galleryLoading !== undefined) {
                            jse.libs.loading_spinner.hide(galleryLoading);
                        }

                        _initializeProductSwiperThumbnail($('#product_thumbnail_swiper'));
                    });

                    gambio.widgets.init($('#product_thumbnail_swiper_mobile')).done(function (arg1, arg2, arg3) {
                        var container = $('#product_thumbnail_swiper_mobile');
                        if (container.swiper() !== undefined) {
                            container.swiper().update();
                        }
                        container.trigger(jse.libs.theme.events.SLIDES_UPDATE());
                        if (galleryLoading !== undefined) {
                            jse.libs.loading_spinner.hide(galleryLoading);
                        }

                        _initializeProductSwiperThumbnail($('#product_thumbnail_swiper_mobile'));
                    });
                    gambio.widgets.init($('.product-info-image')).done(function () {
                        var container = $('.product-info-image');
                        if (container.swiper() !== undefined) {
                            container.swiper().update();
                        }
                        container.trigger(jse.libs.theme.events.SLIDES_UPDATE());
                    });
                    gambio.widgets.init($('#product-info-layer-image')).done(function () {
                        var container = $('#product-info-layer-image');
                        if (container.swiper() !== undefined) {
                            container.swiper().update();
                        }
                        container.trigger(jse.libs.theme.events.SLIDES_UPDATE());
                    });
                    gambio.widgets.init($('#product-info-layer-thumbnails')).done(function () {
                        var container = $('#product-info-layer-thumbnails');
                        if (container.swiper() !== undefined) {
                            container.swiper().update();
                        }
                        container.trigger(jse.libs.theme.events.SLIDES_UPDATE());
                    });
                }
            }).fail(function () {
                _addButtonState($self, '-fail');
            });
        }
    };

    /**
     * Handler for the change property option
     * */
    var _changeProductOptions = function _changeProductOptions(e) {
        var option = e.currentTarget;
        var optionValue = $(option).data('value');
        var optionContainer = $(option).parents('.modifier-group');

        $(optionContainer).find('li.active').removeClass('active');
        $(optionContainer).find('.modifier-item.active-modifier').removeClass('active-modifier');
        $(optionContainer).find('input.hidden-input').val(optionValue);
        $(optionContainer).find('input.hidden-input').trigger('blur', []);

        $(option).parents('li').addClass('active');
        $(option).addClass('active-modifier');
    };

    var _selectSelectedModifierInfo = function _selectSelectedModifierInfo(e) {
        var option = e.currentTarget;
        var price = $(option).attr('data-price');
        var label = $(option).attr('data-label');
        $(option).parents('.modifier-group').find('.selected-value-price').removeClass('temporary-value').attr('data-default-price', price);
        $(option).parents('.modifier-group').find('.selected-value').attr('data-default-value', label);
    };

    var _setSelectedModifierInfo = function _setSelectedModifierInfo(e) {
        var option = e.currentTarget;
        if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
            var price = $(option).attr('data-price');
            var label = $(option).attr('data-label');
            $(option).parents('.modifier-group').find('.selected-value-price').addClass('temporary-value').text(price);
            $(option).parents('.modifier-group').find('.selected-value').text(label);
        }
    };

    var _resetSelectedModifierInfo = function _resetSelectedModifierInfo(e) {
        var option = $(this);
        if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
            var priceHolder = $(option).parents('.modifier-group').find('.selected-value-price');
            var labelHolder = $(option).parents('.modifier-group').find('.selected-value');
            $(priceHolder).removeClass('temporary-value').text($(priceHolder).attr('data-default-price'));
            $(labelHolder).text($(labelHolder).attr('data-default-value'));
        }
    };

    /**
     * Keyup handler for quantity input field
     *
     * @param e
     * @private
     */
    var _keyupHandler = function _keyupHandler(e) {
        clearTimeout(timeout);

        timeout = setTimeout(function () {
            _submitHandler.call(this, e);
        }.bind(this), 300);
    };

    /**
     * Event handler for the add to cart button, that shows or hides the throbber.
     */
    var _addToCartThrobberHandler = function _addToCartThrobberHandler(e) {
        var $btn = $(this);
        var $btnFake = $this.find(".btn-add-to-cart-fake");
        var formReady = true;

        $(".properties-selection-form select").each(function () {
            var val = $(this).val();
            if (!val || val < 1) {
                formReady = false;
            }
        });

        if (formReady) {
            $btn.hide();
            $btnFake.show().prop("disabled", true).prepend('<span class="throbbler"></span>');
        }
    };

    /**
     * Cart dropdown oben event handler for the body.
     */
    var _cartDropdownOpenHandler = function _cartDropdownOpenHandler(e) {
        var $btn = $this.find("[name=btn-add-to-cart]");
        var $btnFake = $this.find(".btn-add-to-cart-fake");
        var fakeOrigLabel = $btnFake.html();
        var productCount = $(".cart-products-count").html();

        var textPhrases = JSON.parse($('#product-details-text-phrases').html());
        console.log(textPhrases['productsInCartSuffix']);

        $btnFake.html("<i class=\"fa fa-check\"></i> " + parseInt(productCount) + textPhrases['productsInCartSuffix']).prop("disabled", true).addClass("btn-buy-complete");

        setTimeout(function () {
            $btnFake.html(fakeOrigLabel).removeClass("btn-buy-complete").hide().prop("disabled", false);
            $(".throbbler", $btn).remove();
            $btn.show();
        }, 5000);
    };

    var _initializeProductSwiperThumbnail = function _initializeProductSwiperThumbnail($element) {
        gambio.widgets.init($element).done(function () {
            if ($element.swiper() !== undefined) {
                $element.swiper().update();
            }
            $element.trigger(jse.libs.theme.events.SLIDES_UPDATE());
        });
    };

    // ########## INITIALIZATION ##########

    /**
     * Init function of the widget
     * @constructor
     */
    module.init = function (done) {

        var $forms = $this.find('form');

        if (options.page === 'product-info') {
            $forms.find("[name=btn-add-to-cart]").on('touchstart touchmove touchend touchcancel', function () {
                return $forms.find("[name=btn-add-to-cart]").click();
            });
            $forms.find("[name=btn-add-to-cart]").on('mouseup', _addToCartThrobberHandler);
            $("body").on('CART_DROPDOWN_OPEN', _cartDropdownOpenHandler);
        }

        $forms.on('submit', { 'target': 'cart' }, _submitHandler).on('click', options.wishlistButtons, { 'target': 'wishlist' }, _submitHandler).on('click', options.priceOfferButtons, { 'target': 'price_offer' }, _submitHandler).on('change', options.attributes, { 'target': 'check' }, _submitHandler).on('mouseover', options.attributes, _setSelectedModifierInfo).on('mouseout', options.attributes, _resetSelectedModifierInfo).on('blur', options.productOptionField, { 'target': 'check' }, _submitHandler).on('click', options.productOptions, { 'target': 'check' }, function (e) {
            _selectSelectedModifierInfo(e);
            _changeProductOptions(e);
        }).on('mouseover', options.productOptions, _setSelectedModifierInfo).on('mouseout', options.productOptions, _resetSelectedModifierInfo).on('blur', options.quantity, { 'target': 'check' }, function (e) {
            _submitHandler(e);
        }).on('keyup', options.quantity, { 'target': 'check' }, _keyupHandler);

        // Fallback if the backend renders incorrect data
        // on initial page call
        $forms.not('.no-status-check').each(function () {
            _submitHandler.call($(this));
        });

        done();
    };

    // Return data to widget engine
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
